import axios from 'axios';
import pLimit from 'p-limit';
const CONCURRENCY_LIMIT = 2; // setting this to any higher value will make cloudflare block you for 5 minutes

const api = axios.create({
  baseURL: 'https://api.cloudflare.com/client/v4',
  headers: {
    "X-Auth-Email": "deez",
    "X-Auth-Key": "peter_griffin",
    "Content-Type": "application/json"
  }
});

async function fetchAllIPAccessRules() {
  const allRules = [];
  let page = 1;
  const perPage = 100;

  while (true) {
    const response = await api.get('/user/firewall/access_rules/rules', {
      params: {
        per_page: perPage,
        page
      }
    });

    const rules = response.data.result;
    const ipRules = rules.filter(rule =>
      rule.configuration.target === 'ip' ||
      rule.configuration.target === 'ip_range' ||
      rule.configuration.target === 'ip6'
    );

    allRules.push(...ipRules);

    const { total_pages } = response.data.result_info;
    console.log(`Fetched page ${page} of ${total_pages} (${ipRules.length} IP rules)`);

    if (page >= total_pages) break;
    page++;
  }

  return allRules;
}

async function deleteRule(ruleId) {
  try {
    await api.delete(`/user/firewall/access_rules/rules/${ruleId}`);
    console.log(`Deleted rule ${ruleId}`);
  } catch (err) {
    console.error(`Failed to delete rule ${ruleId}:`, err.response?.data || err.message);
  }
}

async function deleteAllIPAccessRules() {
  try {
    const ipRules = await fetchAllIPAccessRules();
    console.log(`Total IP access rules to delete: ${ipRules.length}`);

    const limit = pLimit(CONCURRENCY_LIMIT);

    const deletePromises = ipRules.map(rule =>
      limit(() => deleteRule(rule.id))
    );

    await Promise.allSettled(deletePromises);
    console.log('Finished deleting all rules.');
  } catch (err) {
    console.error('Error:', err.response?.data || err.message);
  }
}

deleteAllIPAccessRules();